clear all;

name = 'styrene10';
nr = 13;         % number of differential equations
ktr = 66000;       % transfer rate constant
kt = 1e8;        % termination rate constant
kp = 660;        % propagation rate constant
kd = 1.4e-4;     % dissociation rate constant
f = 0.8;         % initiator efficiency

T = 4e-2;        % initial transfer agent concentration
I = 4e-1;        % initial initiator concentration
M = 3;           % initial monomer concentration
FW = 104.15;

maxci=0.999;     % stop integration at this conversion for initiator I
maxcm=0.99;     % stop integration at this conversion for monomer M

tmax = 1e8;      % alternate time of integration (specify in ode15s function call)

%-----------------------------------------------------------------------------

kvalue=[kd kt ktr kp f kt];
conc=[I M T];
maxc=[maxci maxcm];
y0=zeros(1,nr);
y0(1:3)=conc;

%------do calculation-------------------------------------------------------

tic; % timer on
[t,x]=ode15s('mymoments2',[],y0,[],kvalue,conc,maxc,nr);
toc  % display calculation time

%--------------------------------------------------------------------
%-----------calculate molar mass averages etc------------------------
%--------------------------------------------------------------------
% 1 Mn rad	2 Mw rad	3 PD rad	18 Mn total
% 4 Mn raft	5 Mw raft	6 PD raft	19 Mw total
% 7 Mn dead	8 Mw dead	9 PD dead	20 PD total
% 10 monomer conversion
% 11 initiator conversion
% 12 transfer agent conversion
% 13 wp weight fraction of polymer (equals conversion, calc. indep.)
% 14 weight fraction of dormant chains in the polymer
% 15 weight fraction of dead chains in the polymer
% 16 number fraction of dormant chains in the polymer
% 17 number fraction of dead chains in the polymer
%--------------------------------------------------------------------

r=zeros(max(size(t)),20);
for i=1:max(size(t))
   r(i,1) = (FW*x(i,8))/x(i,6);         % Mn rad  (1st moment / concentration)
   r(i,2) = (FW*x(i,9))/x(i,8);         % Mw rad  (2nd moment / 1st moment)
   r(i,3) = (r(i,2))/(r(i,1));          % PD rad  (Mw / Mn)
   r(i,4) = (FW*x(i,10))/x(i,4);        % Mn raft (1st moment / concentration)
   r(i,5) = (FW*x(i,11))/x(i,10);       % Mw raft (2nd moment / 1st moment)
   r(i,6) = r(i,5)/r(i,4);              % PD raft (Mw / Mn)
   r(i,7) = (FW*x(i,12))/x(i,7);        % Mn dead (1st moment / concentration)
   r(i,8) = (FW*x(i,13))/x(i,12);       % Mw dead (2nd moment / 1st moment)
   r(i,9) = r(i,8)/r(i,7);              % PD dead (Mw / Mn)
   r(i,10) = ((M - x(i,2))/M)*100;      % monomer conversion
   r(i,11) = ((I - x(i,1))/I)*100;      % initiator conversion
   r(i,12) = ((T - x(i,3))/T)*100;      % raft agent conversion
   tempy = x(i,8)+x(i,10)+x(i,12);
   r(i,13) = tempy/(tempy+x(i,2));      % wp
   r(i,14) = x(i,10)/tempy;             % w.fract. dormant in polymer
   r(i,15) = x(i,12)/tempy;             % w.fract. dead in polymer
   tempy = x(i,4)+x(i,6)+x(i,7);
   r(i,16) = x(i,4)/tempy;              % n.fract. dormant in polymer
   r(i,17) = x(i,7)/tempy;              % n.fract. dead in polymer
   totalzero = x(i,4)+x(i,6)+x(i,7);    % summation of zeroth moments of polymers
   totalfrst = x(i,8)+x(i,10)+x(i,12);  % summation of first moments of polymers
   totalscnd = x(i,9)+x(i,11)+x(i,13);  % summation of first moments of polymers
   r(i,18) = FW*totalfrst/totalzero;       % Mn total
   r(i,19) = FW*totalscnd/totalfrst;       % Mw total
   r(i,20) = r(i,19)/r(i,18);           % PD total

end
clear tempy


%---------output-file-1------------------------------------------------------

temp = [name '_SIM.dat'];
fm = fopen(temp,'w');
head ='time\tmc\tini\tmono\traft\tdorm\tr\trad\tdead\tm1rad\t';
head =[head 'm2rad\tm1raft\tm2raft\tm1dead\tm2dead\n'];

fprintf(fm,head);

for i=1:max(size(t))
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\t',t(i),r(i,10),x(i,1),x(i,2),x(i,3),x(i,4));
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\t',x(i,5),x(i,6),x(i,7),x(i,8),x(i,9));
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\n',x(i,10),x(i,11),x(i,12),x(i,13));
end

fclose(fm);

clear temp;
clear head;

%---------output-file-2------------------------------------------------------

temp = [name '_M.dat'];
fm = fopen(temp,'w');
head ='time\tmc\tic\ttc\tmnrad\tmwrad\tpdrad\t';
head =[head 'mnraft\tmwraft\tpdraft\tmndead\tmwdead\tpddead\t'];
head =[head 'mntot\tmwtot\tpdtot\n'];

fprintf(fm,head);

for i=1:max(size(t))
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\t',t(i),r(i,10),r(i,11),r(i,12));
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\t',r(i,1),r(i,2),r(i,3));
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\t',r(i,4),r(i,5),r(i,6));
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\t',r(i,7),r(i,8),r(i,9));
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\n',r(i,18),r(i,19),r(i,20));
end

fclose(fm);

clear temp;
clear head;

%---------output-file-3------------------------------------------------------

temp = [name '_wp.dat'];
fm = fopen(temp,'w');
head ='time\tmc\twp\twfdorm\twfdead\tnfdorm\tnfdead\n';

fprintf(fm,head);

for i=1:max(size(t))
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\t%+4.3e\t',t(i),r(i,10),r(i,13),r(i,14));
  fprintf(fm,'%+4.3e\t%+4.3e\t%+4.3e\n',r(i,15),r(i,16),r(i,17));
end

fclose(fm);

clear temp;
clear head;

%---------output-file-4------------------------------------------------------

temp = [name '.txt'];
fm = fopen(temp,'w');
fprintf(fm, 'simulation: %s\r\n', name);
fprintf(fm, 'ktr = %+4.3g\r\n', ktr);
fprintf(fm, 'kt  = %+4.3g\r\n', kt);
fprintf(fm, 'kp  = %+4.3g\r\n', kp);
fprintf(fm, 'kd  = %+4.3g\r\n', kd);
fprintf(fm, 'f   = %+4.3g (ini.efficiency)\r\n', f);
fprintf(fm, '\r\n');
fprintf(fm, 'RAFT= %+4.3g\r\n', T);
fprintf(fm, 'INI = %+4.3g\r\n', I);
fprintf(fm, 'M   = %+4.3g\r\n', M);
fprintf(fm, 'FW  = %+4.3g (g/mol)\r\n', FW);
fclose(fm);

clear all;